const { ipcRenderer } = require('electron');

// ===== Configuration & State =====
const CONFIG = {
  defaultWidth: 32,
  defaultHeight: 32,
  maxUndoSteps: 50,
  minZoom: 1,
  maxZoom: 64,
  defaultZoom: 12
};

const state = {
  canvasWidth: CONFIG.defaultWidth,
  canvasHeight: CONFIG.defaultHeight,
  zoom: CONFIG.defaultZoom,
  currentTool: 'pencil',
  brushSize: 1,
  primaryColor: { r: 0, g: 0, b: 0, a: 255 },
  secondaryColor: { r: 255, g: 255, b: 255, a: 255 },
  showGrid: true,
  isDrawing: false,
  lastX: -1,
  lastY: -1,
  undoStack: [],
  redoStack: [],
  layers: [],
  activeLayerIndex: 0,
  selection: null,
  selectionData: null,
  shapeStart: null,
  modified: false
};

// ===== Palette Presets =====
const PALETTES = {
  default: [
    '#000000', '#ffffff', '#ff0000', '#00ff00', '#0000ff', '#ffff00',
    '#ff00ff', '#00ffff', '#ff8000', '#8000ff', '#0080ff', '#ff0080',
    '#808080', '#c0c0c0', '#800000', '#008000', '#000080', '#808000',
    '#800080', '#008080', '#400000', '#004000', '#000040', '#404040'
  ],
  gameboy: ['#0f380f', '#306230', '#8bac0f', '#9bbc0f'],
  nes: [
    '#000000', '#fcfcfc', '#f8f8f8', '#bcbcbc', '#7c7c7c', '#a4e4fc',
    '#3cbcfc', '#0078f8', '#0000fc', '#b8b8f8', '#6888fc', '#0058f8',
    '#0000bc', '#d8b8f8', '#9878f8', '#6844fc', '#4428bc', '#f8b8f8',
    '#f878f8', '#d800cc', '#940084', '#f8a4c0', '#f85898', '#e40058',
    '#a80020', '#f0d0b0', '#f87858', '#f83800', '#a81000', '#fce0a8',
    '#fca044', '#e45c10', '#881400', '#f8d878', '#f8b800', '#ac7c00',
    '#503000', '#d8f878', '#b8f818', '#00b800', '#007800', '#b8f8b8',
    '#58d854', '#00a800', '#006800', '#b8f8d8', '#58f898', '#00a844',
    '#005800', '#00fcfc', '#00e8d8', '#008888', '#004058', '#f8d8f8',
    '#787878'
  ],
  commodore64: [
    '#000000', '#ffffff', '#880000', '#aaffee', '#cc44cc', '#00cc55',
    '#0000aa', '#eeee77', '#dd8855', '#664400', '#ff7777', '#333333',
    '#777777', '#aaff66', '#0088ff', '#bbbbbb'
  ],
  pico8: [
    '#000000', '#1d2b53', '#7e2553', '#008751', '#ab5236', '#5f574f',
    '#c2c3c7', '#fff1e8', '#ff004d', '#ffa300', '#ffec27', '#00e436',
    '#29adff', '#83769c', '#ff77a8', '#ffccaa'
  ],
  sweetie16: [
    '#1a1c2c', '#5d275d', '#b13e53', '#ef7d57', '#ffcd75', '#a7f070',
    '#38b764', '#257179', '#29366f', '#3b5dc9', '#41a6f6', '#73eff7',
    '#f4f4f4', '#94b0c2', '#566c86', '#333c57'
  ],
  endesga32: [
    '#be4a2f', '#d77643', '#ead4aa', '#e4a672', '#b86f50', '#733e39',
    '#3e2731', '#a22633', '#e43b44', '#f77622', '#feae34', '#fee761',
    '#63c74d', '#3e8948', '#265c42', '#193c3e', '#124e89', '#0099db',
    '#2ce8f5', '#ffffff', '#c0cbdc', '#8b9bb4', '#5a6988', '#3a4466',
    '#262b44', '#181425', '#ff0044', '#68386c', '#b55088', '#f6757a',
    '#e8b796', '#c28569'
  ]
};

// ===== DOM Elements =====
const elements = {
  pixelCanvas: document.getElementById('pixel-canvas'),
  gridCanvas: document.getElementById('grid-canvas'),
  previewCanvas: document.getElementById('preview-canvas'),
  canvasWrapper: document.getElementById('canvas-wrapper'),
  canvasContainer: document.getElementById('canvas-container'),
  layersContainer: document.getElementById('layers-container'),
  paletteGrid: document.getElementById('palette-grid'),
  colorPicker: document.getElementById('color-picker'),
  hexInput: document.getElementById('hex-input'),
  rInput: document.getElementById('r-input'),
  gInput: document.getElementById('g-input'),
  bInput: document.getElementById('b-input'),
  alphaInput: document.getElementById('alpha-input'),
  alphaDisplay: document.getElementById('alpha-display'),
  primaryColor: document.getElementById('primary-color'),
  secondaryColor: document.getElementById('secondary-color'),
  brushSize: document.getElementById('brush-size'),
  brushSizeDisplay: document.getElementById('brush-size-display'),
  zoomDisplay: document.getElementById('zoom-display'),
  statusCoords: document.getElementById('status-coords'),
  statusCanvasSize: document.getElementById('status-canvas-size'),
  statusTool: document.getElementById('status-tool'),
  statusLayer: document.getElementById('status-layer'),
  canvasWidth: document.getElementById('canvas-width'),
  canvasHeight: document.getElementById('canvas-height'),
  palettePresets: document.getElementById('palette-presets')
};

const ctx = elements.pixelCanvas.getContext('2d', { willReadFrequently: true });
const gridCtx = elements.gridCanvas.getContext('2d');
const previewCtx = elements.previewCanvas.getContext('2d');

// ===== Initialization =====
function init() {
  setupCanvas(CONFIG.defaultWidth, CONFIG.defaultHeight);
  addLayer('Background');
  setupEventListeners();
  updateColorDisplay();
  loadPalette('default');
  updateZoomDisplay();
  updateStatusBar();
  selectTool('pencil');
}

function setupCanvas(width, height) {
  state.canvasWidth = width;
  state.canvasHeight = height;

  elements.pixelCanvas.width = width;
  elements.pixelCanvas.height = height;
  elements.gridCanvas.width = width;
  elements.gridCanvas.height = height;
  elements.previewCanvas.width = width;
  elements.previewCanvas.height = height;

  updateCanvasSize();
  drawGrid();

  elements.canvasWidth.value = width;
  elements.canvasHeight.value = height;
  elements.statusCanvasSize.textContent = `${width} × ${height}`;
}

function updateCanvasSize() {
  const displayWidth = state.canvasWidth * state.zoom;
  const displayHeight = state.canvasHeight * state.zoom;

  elements.canvasWrapper.style.width = `${displayWidth}px`;
  elements.canvasWrapper.style.height = `${displayHeight}px`;

  // Pixel and preview canvases use CSS scaling for pixelated look
  [elements.pixelCanvas, elements.previewCanvas].forEach(canvas => {
    canvas.style.width = `${displayWidth}px`;
    canvas.style.height = `${displayHeight}px`;
  });

  // Grid canvas is handled separately in drawGrid() at display resolution
}

// ===== Layer Management =====
function addLayer(name = null) {
  const layerIndex = state.layers.length;
  const layerName = name || `Layer ${layerIndex + 1}`;

  const layerCanvas = document.createElement('canvas');
  layerCanvas.width = state.canvasWidth;
  layerCanvas.height = state.canvasHeight;

  const layer = {
    name: layerName,
    canvas: layerCanvas,
    ctx: layerCanvas.getContext('2d', { willReadFrequently: true }),
    visible: true,
    opacity: 1
  };

  state.layers.push(layer);
  state.activeLayerIndex = layerIndex;
  renderLayersList();
  compositeCanvas();
}

function deleteLayer() {
  if (state.layers.length <= 1) return;

  state.layers.splice(state.activeLayerIndex, 1);
  if (state.activeLayerIndex >= state.layers.length) {
    state.activeLayerIndex = state.layers.length - 1;
  }
  renderLayersList();
  compositeCanvas();
}

function setActiveLayer(index) {
  state.activeLayerIndex = index;
  renderLayersList();
  updateStatusBar();
}

function toggleLayerVisibility(index) {
  state.layers[index].visible = !state.layers[index].visible;
  renderLayersList();
  compositeCanvas();
}

function renderLayersList() {
  elements.layersContainer.innerHTML = '';

  for (let i = state.layers.length - 1; i >= 0; i--) {
    const layer = state.layers[i];
    const layerEl = document.createElement('div');
    layerEl.className = `layer-item ${i === state.activeLayerIndex ? 'active' : ''}`;
    layerEl.dataset.index = i;

    layerEl.innerHTML = `
      <div class="layer-visibility" data-index="${i}">
        <svg viewBox="0 0 24 24">${layer.visible ?
      '<path d="M12 4.5C7 4.5 2.73 7.61 1 12c1.73 4.39 6 7.5 11 7.5s9.27-3.11 11-7.5c-1.73-4.39-6-7.5-11-7.5zM12 17c-2.76 0-5-2.24-5-5s2.24-5 5-5 5 2.24 5 5-2.24 5-5 5zm0-8c-1.66 0-3 1.34-3 3s1.34 3 3 3 3-1.34 3-3-1.34-3-3-3z"/>' :
      '<path d="M12 7c2.76 0 5 2.24 5 5 0 .65-.13 1.26-.36 1.83l2.92 2.92c1.51-1.26 2.7-2.89 3.43-4.75-1.73-4.39-6-7.5-11-7.5-1.4 0-2.74.25-3.98.7l2.16 2.16C10.74 7.13 11.35 7 12 7zM2 4.27l2.28 2.28.46.46C3.08 8.3 1.78 10.02 1 12c1.73 4.39 6 7.5 11 7.5 1.55 0 3.03-.3 4.38-.84l.42.42L19.73 22 21 20.73 3.27 3 2 4.27zM7.53 9.8l1.55 1.55c-.05.21-.08.43-.08.65 0 1.66 1.34 3 3 3 .22 0 .44-.03.65-.08l1.55 1.55c-.67.33-1.41.53-2.2.53-2.76 0-5-2.24-5-5 0-.79.2-1.53.53-2.2zm4.31-.78l3.15 3.15.02-.16c0-1.66-1.34-3-3-3l-.17.01z"/>'
    }</svg>
      </div>
      <div class="layer-preview">
        <canvas width="32" height="32"></canvas>
      </div>
      <div class="layer-name">
        <input type="text" value="${layer.name}" data-index="${i}">
      </div>
    `;

    // Update preview
    const previewCanvas = layerEl.querySelector('.layer-preview canvas');
    const previewCtx = previewCanvas.getContext('2d');
    previewCtx.drawImage(layer.canvas, 0, 0, 32, 32);

    elements.layersContainer.appendChild(layerEl);
  }

  // Event listeners for layer items
  document.querySelectorAll('.layer-item').forEach(el => {
    el.addEventListener('click', (e) => {
      if (!e.target.closest('.layer-visibility') && !e.target.closest('input')) {
        setActiveLayer(parseInt(el.dataset.index));
      }
    });
  });

  document.querySelectorAll('.layer-visibility').forEach(el => {
    el.addEventListener('click', () => {
      toggleLayerVisibility(parseInt(el.dataset.index));
    });
  });

  document.querySelectorAll('.layer-name input').forEach(el => {
    el.addEventListener('change', () => {
      state.layers[parseInt(el.dataset.index)].name = el.value;
    });
    el.addEventListener('click', (e) => e.stopPropagation());
  });

  updateStatusBar();
}

function compositeCanvas() {
  ctx.clearRect(0, 0, state.canvasWidth, state.canvasHeight);

  for (const layer of state.layers) {
    if (layer.visible) {
      ctx.globalAlpha = layer.opacity;
      ctx.drawImage(layer.canvas, 0, 0);
    }
  }
  ctx.globalAlpha = 1;
}

// ===== Drawing Tools =====
function getPixelCoords(e) {
  const rect = elements.pixelCanvas.getBoundingClientRect();
  const x = Math.floor((e.clientX - rect.left) / state.zoom);
  const y = Math.floor((e.clientY - rect.top) / state.zoom);
  return { x, y };
}

function isInBounds(x, y) {
  return x >= 0 && x < state.canvasWidth && y >= 0 && y < state.canvasHeight;
}

function colorToRgba(color) {
  return `rgba(${color.r}, ${color.g}, ${color.b}, ${color.a / 255})`;
}

function rgbaToHex(r, g, b) {
  return '#' + [r, g, b].map(x => x.toString(16).padStart(2, '0')).join('');
}

function hexToRgb(hex) {
  const result = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(hex);
  return result ? {
    r: parseInt(result[1], 16),
    g: parseInt(result[2], 16),
    b: parseInt(result[3], 16)
  } : null;
}

function setPixel(layerCtx, x, y, color) {
  layerCtx.fillStyle = colorToRgba(color);
  layerCtx.fillRect(x, y, 1, 1);
}

function erasePixel(layerCtx, x, y) {
  layerCtx.clearRect(x, y, 1, 1);
}

function drawBrush(layerCtx, x, y, color, erase = false) {
  const size = state.brushSize;
  const offset = Math.floor(size / 2);

  for (let dy = 0; dy < size; dy++) {
    for (let dx = 0; dx < size; dx++) {
      const px = x - offset + dx;
      const py = y - offset + dy;
      if (isInBounds(px, py)) {
        if (erase) {
          erasePixel(layerCtx, px, py);
        } else {
          setPixel(layerCtx, px, py, color);
        }
      }
    }
  }
}

function drawLine(layerCtx, x0, y0, x1, y1, color, erase = false) {
  const dx = Math.abs(x1 - x0);
  const dy = Math.abs(y1 - y0);
  const sx = x0 < x1 ? 1 : -1;
  const sy = y0 < y1 ? 1 : -1;
  let err = dx - dy;

  while (true) {
    drawBrush(layerCtx, x0, y0, color, erase);

    if (x0 === x1 && y0 === y1) break;
    const e2 = 2 * err;
    if (e2 > -dy) {
      err -= dy;
      x0 += sx;
    }
    if (e2 < dx) {
      err += dx;
      y0 += sy;
    }
  }
}

function floodFill(x, y, newColor) {
  const layer = state.layers[state.activeLayerIndex];
  const imageData = layer.ctx.getImageData(0, 0, state.canvasWidth, state.canvasHeight);
  const data = imageData.data;

  const targetIdx = (y * state.canvasWidth + x) * 4;
  const targetColor = {
    r: data[targetIdx],
    g: data[targetIdx + 1],
    b: data[targetIdx + 2],
    a: data[targetIdx + 3]
  };

  if (targetColor.r === newColor.r && targetColor.g === newColor.g &&
    targetColor.b === newColor.b && targetColor.a === newColor.a) {
    return;
  }

  const stack = [[x, y]];
  const visited = new Set();

  while (stack.length > 0) {
    const [cx, cy] = stack.pop();
    const key = `${cx},${cy}`;

    if (visited.has(key) || !isInBounds(cx, cy)) continue;

    const idx = (cy * state.canvasWidth + cx) * 4;
    if (data[idx] !== targetColor.r || data[idx + 1] !== targetColor.g ||
      data[idx + 2] !== targetColor.b || data[idx + 3] !== targetColor.a) {
      continue;
    }

    visited.add(key);
    data[idx] = newColor.r;
    data[idx + 1] = newColor.g;
    data[idx + 2] = newColor.b;
    data[idx + 3] = newColor.a;

    stack.push([cx + 1, cy], [cx - 1, cy], [cx, cy + 1], [cx, cy - 1]);
  }

  layer.ctx.putImageData(imageData, 0, 0);
}

function eyedropper(x, y) {
  const imageData = ctx.getImageData(x, y, 1, 1).data;
  state.primaryColor = {
    r: imageData[0],
    g: imageData[1],
    b: imageData[2],
    a: imageData[3]
  };
  updateColorDisplay();
}

function drawRectangle(ctx, x0, y0, x1, y1, color, filled = false) {
  const minX = Math.min(x0, x1);
  const minY = Math.min(y0, y1);
  const maxX = Math.max(x0, x1);
  const maxY = Math.max(y0, y1);

  ctx.fillStyle = colorToRgba(color);

  if (filled) {
    for (let y = minY; y <= maxY; y++) {
      for (let x = minX; x <= maxX; x++) {
        if (isInBounds(x, y)) {
          ctx.fillRect(x, y, 1, 1);
        }
      }
    }
  } else {
    for (let x = minX; x <= maxX; x++) {
      if (isInBounds(x, minY)) ctx.fillRect(x, minY, 1, 1);
      if (isInBounds(x, maxY)) ctx.fillRect(x, maxY, 1, 1);
    }
    for (let y = minY + 1; y < maxY; y++) {
      if (isInBounds(minX, y)) ctx.fillRect(minX, y, 1, 1);
      if (isInBounds(maxX, y)) ctx.fillRect(maxX, y, 1, 1);
    }
  }
}

function drawEllipse(ctx, x0, y0, x1, y1, color, filled = false) {
  const cx = (x0 + x1) / 2;
  const cy = (y0 + y1) / 2;
  const rx = Math.abs(x1 - x0) / 2;
  const ry = Math.abs(y1 - y0) / 2;

  ctx.fillStyle = colorToRgba(color);

  if (filled) {
    for (let y = Math.floor(cy - ry); y <= Math.ceil(cy + ry); y++) {
      for (let x = Math.floor(cx - rx); x <= Math.ceil(cx + rx); x++) {
        const dx = (x - cx) / (rx || 1);
        const dy = (y - cy) / (ry || 1);
        if (dx * dx + dy * dy <= 1 && isInBounds(x, y)) {
          ctx.fillRect(x, y, 1, 1);
        }
      }
    }
  } else {
    // Midpoint ellipse algorithm for outline
    let x = 0;
    let y = Math.round(ry);
    const rxSq = rx * rx;
    const rySq = ry * ry;

    const plotPoints = (px, py) => {
      const points = [
        [Math.round(cx + px), Math.round(cy + py)],
        [Math.round(cx - px), Math.round(cy + py)],
        [Math.round(cx + px), Math.round(cy - py)],
        [Math.round(cx - px), Math.round(cy - py)]
      ];
      points.forEach(([x, y]) => {
        if (isInBounds(x, y)) ctx.fillRect(x, y, 1, 1);
      });
    };

    let p1 = rySq - rxSq * ry + 0.25 * rxSq;
    while (rySq * x < rxSq * y) {
      plotPoints(x, y);
      x++;
      if (p1 < 0) {
        p1 += 2 * rySq * x + rySq;
      } else {
        y--;
        p1 += 2 * rySq * x - 2 * rxSq * y + rySq;
      }
    }

    let p2 = rySq * (x + 0.5) * (x + 0.5) + rxSq * (y - 1) * (y - 1) - rxSq * rySq;
    while (y >= 0) {
      plotPoints(x, y);
      y--;
      if (p2 > 0) {
        p2 += rxSq - 2 * rxSq * y;
      } else {
        x++;
        p2 += 2 * rySq * x - 2 * rxSq * y + rxSq;
      }
    }
  }
}

// ===== Preview Drawing =====
function clearPreview() {
  previewCtx.clearRect(0, 0, state.canvasWidth, state.canvasHeight);
}

function drawPreview(x, y) {
  clearPreview();

  if (!state.shapeStart) return;

  const color = { ...state.primaryColor, a: 128 };

  switch (state.currentTool) {
    case 'line':
      drawLine(previewCtx, state.shapeStart.x, state.shapeStart.y, x, y, color);
      break;
    case 'rectangle':
      drawRectangle(previewCtx, state.shapeStart.x, state.shapeStart.y, x, y, color);
      break;
    case 'ellipse':
      drawEllipse(previewCtx, state.shapeStart.x, state.shapeStart.y, x, y, color);
      break;
  }
}

// ===== Selection Tool =====
function startSelection(x, y) {
  state.selection = { x0: x, y0: y, x1: x, y1: y };
}

function updateSelection(x, y) {
  if (state.selection) {
    state.selection.x1 = x;
    state.selection.y1 = y;
    drawSelectionOverlay();
  }
}

function drawSelectionOverlay() {
  clearPreview();
  if (!state.selection) return;

  const { x0, y0, x1, y1 } = state.selection;
  const minX = Math.min(x0, x1);
  const minY = Math.min(y0, y1);
  const width = Math.abs(x1 - x0) + 1;
  const height = Math.abs(y1 - y0) + 1;

  previewCtx.strokeStyle = '#6366f1';
  previewCtx.lineWidth = 1 / state.zoom;
  previewCtx.setLineDash([2 / state.zoom, 2 / state.zoom]);
  previewCtx.strokeRect(minX, minY, width, height);
  previewCtx.setLineDash([]);
}

function copySelection() {
  if (!state.selection) return;

  const { x0, y0, x1, y1 } = state.selection;
  const minX = Math.min(x0, x1);
  const minY = Math.min(y0, y1);
  const width = Math.abs(x1 - x0) + 1;
  const height = Math.abs(y1 - y0) + 1;

  const layer = state.layers[state.activeLayerIndex];
  state.selectionData = {
    imageData: layer.ctx.getImageData(minX, minY, width, height),
    x: minX,
    y: minY,
    width,
    height
  };
}

function deleteSelection() {
  if (!state.selection) return;

  saveState();
  const { x0, y0, x1, y1 } = state.selection;
  const minX = Math.min(x0, x1);
  const minY = Math.min(y0, y1);
  const width = Math.abs(x1 - x0) + 1;
  const height = Math.abs(y1 - y0) + 1;

  const layer = state.layers[state.activeLayerIndex];
  layer.ctx.clearRect(minX, minY, width, height);
  compositeCanvas();
  renderLayersList();
}

// ===== Undo/Redo =====
function saveState() {
  const layersData = state.layers.map(layer => ({
    name: layer.name,
    imageData: layer.ctx.getImageData(0, 0, state.canvasWidth, state.canvasHeight),
    visible: layer.visible,
    opacity: layer.opacity
  }));

  state.undoStack.push({
    layers: layersData,
    activeLayerIndex: state.activeLayerIndex
  });

  if (state.undoStack.length > CONFIG.maxUndoSteps) {
    state.undoStack.shift();
  }

  state.redoStack = [];
  state.modified = true;
}

function undo() {
  if (state.undoStack.length === 0) return;

  const currentState = {
    layers: state.layers.map(layer => ({
      name: layer.name,
      imageData: layer.ctx.getImageData(0, 0, state.canvasWidth, state.canvasHeight),
      visible: layer.visible,
      opacity: layer.opacity
    })),
    activeLayerIndex: state.activeLayerIndex
  };
  state.redoStack.push(currentState);

  const prevState = state.undoStack.pop();
  restoreState(prevState);
}

function redo() {
  if (state.redoStack.length === 0) return;

  const currentState = {
    layers: state.layers.map(layer => ({
      name: layer.name,
      imageData: layer.ctx.getImageData(0, 0, state.canvasWidth, state.canvasHeight),
      visible: layer.visible,
      opacity: layer.opacity
    })),
    activeLayerIndex: state.activeLayerIndex
  };
  state.undoStack.push(currentState);

  const nextState = state.redoStack.pop();
  restoreState(nextState);
}

function restoreState(savedState) {
  state.layers = savedState.layers.map(layerData => {
    const canvas = document.createElement('canvas');
    canvas.width = state.canvasWidth;
    canvas.height = state.canvasHeight;
    const ctx = canvas.getContext('2d', { willReadFrequently: true });
    ctx.putImageData(layerData.imageData, 0, 0);

    return {
      name: layerData.name,
      canvas,
      ctx,
      visible: layerData.visible,
      opacity: layerData.opacity
    };
  });

  state.activeLayerIndex = savedState.activeLayerIndex;
  renderLayersList();
  compositeCanvas();
}

// ===== Grid =====
function drawGrid() {
  // Resize grid canvas to display size for crisp lines
  const displayWidth = state.canvasWidth * state.zoom;
  const displayHeight = state.canvasHeight * state.zoom;

  elements.gridCanvas.width = displayWidth;
  elements.gridCanvas.height = displayHeight;
  elements.gridCanvas.style.width = `${displayWidth}px`;
  elements.gridCanvas.style.height = `${displayHeight}px`;

  gridCtx.clearRect(0, 0, displayWidth, displayHeight);

  if (!state.showGrid || state.zoom < 4) return;

  gridCtx.strokeStyle = 'rgba(255, 255, 255, 0.3)';
  gridCtx.lineWidth = 1;

  // Draw vertical lines
  for (let x = 0; x <= state.canvasWidth; x++) {
    const px = x * state.zoom;
    gridCtx.beginPath();
    gridCtx.moveTo(px + 0.5, 0);
    gridCtx.lineTo(px + 0.5, displayHeight);
    gridCtx.stroke();
  }

  // Draw horizontal lines
  for (let y = 0; y <= state.canvasHeight; y++) {
    const py = y * state.zoom;
    gridCtx.beginPath();
    gridCtx.moveTo(0, py + 0.5);
    gridCtx.lineTo(displayWidth, py + 0.5);
    gridCtx.stroke();
  }
}

function toggleGrid() {
  state.showGrid = !state.showGrid;
  drawGrid();
  document.getElementById('grid-btn').classList.toggle('active', state.showGrid);
}

// ===== Zoom =====
function setZoom(newZoom) {
  state.zoom = Math.max(CONFIG.minZoom, Math.min(CONFIG.maxZoom, newZoom));
  updateCanvasSize();
  drawGrid();
  updateZoomDisplay();
}

function zoomIn() {
  setZoom(state.zoom * 2);
}

function zoomOut() {
  setZoom(state.zoom / 2);
}

function resetZoom() {
  setZoom(CONFIG.defaultZoom);
}

function updateZoomDisplay() {
  elements.zoomDisplay.textContent = `${Math.round(state.zoom * 100 / CONFIG.defaultZoom * 100)}%`;
}

// ===== Color Management =====
function updateColorDisplay() {
  const color = state.primaryColor;
  elements.primaryColor.style.background = colorToRgba(color);
  elements.secondaryColor.style.background = colorToRgba(state.secondaryColor);
  elements.colorPicker.value = rgbaToHex(color.r, color.g, color.b);
  elements.hexInput.value = rgbaToHex(color.r, color.g, color.b);
  elements.rInput.value = color.r;
  elements.gInput.value = color.g;
  elements.bInput.value = color.b;
  elements.alphaInput.value = color.a;
  elements.alphaDisplay.textContent = color.a;
}

function swapColors() {
  const temp = state.primaryColor;
  state.primaryColor = state.secondaryColor;
  state.secondaryColor = temp;
  updateColorDisplay();
}

function loadPalette(paletteName) {
  const colors = PALETTES[paletteName] || PALETTES.default;
  elements.paletteGrid.innerHTML = '';

  colors.forEach(color => {
    const colorEl = document.createElement('div');
    colorEl.className = 'palette-color';
    colorEl.style.background = color;
    colorEl.dataset.color = color;
    colorEl.addEventListener('click', () => {
      const rgb = hexToRgb(color);
      if (rgb) {
        state.primaryColor = { ...rgb, a: state.primaryColor.a };
        updateColorDisplay();
      }
    });
    colorEl.addEventListener('contextmenu', (e) => {
      e.preventDefault();
      const rgb = hexToRgb(color);
      if (rgb) {
        state.secondaryColor = { ...rgb, a: 255 };
        updateColorDisplay();
      }
    });
    elements.paletteGrid.appendChild(colorEl);
  });
}

function addToPalette() {
  const color = rgbaToHex(state.primaryColor.r, state.primaryColor.g, state.primaryColor.b);
  const colorEl = document.createElement('div');
  colorEl.className = 'palette-color';
  colorEl.style.background = color;
  colorEl.dataset.color = color;
  colorEl.addEventListener('click', () => {
    const rgb = hexToRgb(color);
    if (rgb) {
      state.primaryColor = { ...rgb, a: state.primaryColor.a };
      updateColorDisplay();
    }
  });
  elements.paletteGrid.appendChild(colorEl);
}

// ===== Tool Selection =====
function selectTool(tool) {
  state.currentTool = tool;
  document.querySelectorAll('.tool-btn').forEach(btn => {
    btn.classList.toggle('active', btn.dataset.tool === tool);
  });
  elements.statusTool.textContent = tool.charAt(0).toUpperCase() + tool.slice(1);

  // Clear selection when switching tools
  if (tool !== 'select' && tool !== 'move') {
    state.selection = null;
    clearPreview();
  }
}

// ===== Status Bar =====
function updateStatusBar() {
  elements.statusLayer.textContent = state.layers[state.activeLayerIndex]?.name || 'Layer 1';
}

// ===== Export =====
async function exportPNG() {
  const exportCanvas = document.createElement('canvas');
  exportCanvas.width = state.canvasWidth;
  exportCanvas.height = state.canvasHeight;
  const exportCtx = exportCanvas.getContext('2d');

  for (const layer of state.layers) {
    if (layer.visible) {
      exportCtx.globalAlpha = layer.opacity;
      exportCtx.drawImage(layer.canvas, 0, 0);
    }
  }

  const dataUrl = exportCanvas.toDataURL('image/png');
  const result = await ipcRenderer.invoke('save-png', {
    dataUrl,
    defaultName: `pixel-art-${Date.now()}.png`
  });

  if (result.success) {
    console.log('Saved to:', result.path);
  }
}

async function saveProject() {
  const projectData = {
    version: '1.0',
    canvasWidth: state.canvasWidth,
    canvasHeight: state.canvasHeight,
    layers: state.layers.map(layer => ({
      name: layer.name,
      visible: layer.visible,
      opacity: layer.opacity,
      data: layer.canvas.toDataURL()
    })),
    activeLayerIndex: state.activeLayerIndex,
    palette: Array.from(elements.paletteGrid.querySelectorAll('.palette-color')).map(el => el.dataset.color)
  };

  const result = await ipcRenderer.invoke('save-project', projectData);
  if (result.success) {
    state.modified = false;
  }
}

// ===== New Canvas =====
function showNewCanvasModal() {
  document.getElementById('new-canvas-modal').classList.add('active');
}

function hideNewCanvasModal() {
  document.getElementById('new-canvas-modal').classList.remove('active');
}

function createNewCanvas() {
  const width = parseInt(document.getElementById('new-width').value) || 32;
  const height = parseInt(document.getElementById('new-height').value) || 32;
  const bgOption = document.getElementById('new-bg').value;
  const bgColor = document.getElementById('new-bg-color').value;

  state.layers = [];
  state.undoStack = [];
  state.redoStack = [];
  state.selection = null;

  setupCanvas(width, height);
  addLayer('Background');

  const layer = state.layers[0];
  if (bgOption === 'white') {
    layer.ctx.fillStyle = '#ffffff';
    layer.ctx.fillRect(0, 0, width, height);
  } else if (bgOption === 'black') {
    layer.ctx.fillStyle = '#000000';
    layer.ctx.fillRect(0, 0, width, height);
  } else if (bgOption === 'custom') {
    layer.ctx.fillStyle = bgColor;
    layer.ctx.fillRect(0, 0, width, height);
  }

  compositeCanvas();
  hideNewCanvasModal();
}

function resizeCanvas() {
  const newWidth = parseInt(elements.canvasWidth.value) || state.canvasWidth;
  const newHeight = parseInt(elements.canvasHeight.value) || state.canvasHeight;

  if (newWidth === state.canvasWidth && newHeight === state.canvasHeight) return;

  saveState();

  // Save current layer data
  const layersData = state.layers.map(layer => ({
    imageData: layer.ctx.getImageData(0, 0, state.canvasWidth, state.canvasHeight),
    name: layer.name,
    visible: layer.visible,
    opacity: layer.opacity
  }));

  // Resize
  state.canvasWidth = newWidth;
  state.canvasHeight = newHeight;

  elements.pixelCanvas.width = newWidth;
  elements.pixelCanvas.height = newHeight;
  elements.gridCanvas.width = newWidth;
  elements.gridCanvas.height = newHeight;
  elements.previewCanvas.width = newWidth;
  elements.previewCanvas.height = newHeight;

  // Restore layers
  state.layers = layersData.map(data => {
    const canvas = document.createElement('canvas');
    canvas.width = newWidth;
    canvas.height = newHeight;
    const ctx = canvas.getContext('2d', { willReadFrequently: true });
    ctx.putImageData(data.imageData, 0, 0);

    return {
      name: data.name,
      canvas,
      ctx,
      visible: data.visible,
      opacity: data.opacity
    };
  });

  updateCanvasSize();
  drawGrid();
  compositeCanvas();
  renderLayersList();
  elements.statusCanvasSize.textContent = `${newWidth} × ${newHeight}`;
}

// ===== Event Listeners =====
function setupEventListeners() {
  // Canvas mouse events
  elements.canvasContainer.addEventListener('mousedown', handleMouseDown);
  elements.canvasContainer.addEventListener('mousemove', handleMouseMove);
  elements.canvasContainer.addEventListener('mouseup', handleMouseUp);
  elements.canvasContainer.addEventListener('mouseleave', handleMouseUp);
  elements.canvasContainer.addEventListener('contextmenu', (e) => e.preventDefault());
  elements.canvasContainer.addEventListener('wheel', handleWheel, { passive: false });

  // Tool buttons
  document.querySelectorAll('.tool-btn').forEach(btn => {
    btn.addEventListener('click', () => selectTool(btn.dataset.tool));
  });

  // Action buttons
  document.getElementById('undo-btn').addEventListener('click', undo);
  document.getElementById('redo-btn').addEventListener('click', redo);
  document.getElementById('zoom-in-btn').addEventListener('click', zoomIn);
  document.getElementById('zoom-out-btn').addEventListener('click', zoomOut);
  document.getElementById('grid-btn').addEventListener('click', toggleGrid);
  document.getElementById('export-btn').addEventListener('click', exportPNG);
  document.getElementById('add-layer-btn').addEventListener('click', () => addLayer());
  document.getElementById('delete-layer-btn').addEventListener('click', deleteLayer);
  document.getElementById('swap-colors').addEventListener('click', swapColors);
  document.getElementById('add-to-palette').addEventListener('click', addToPalette);
  document.getElementById('clear-palette').addEventListener('click', () => {
    elements.paletteGrid.innerHTML = '';
  });
  document.getElementById('resize-canvas-btn').addEventListener('click', resizeCanvas);

  // Color inputs
  elements.colorPicker.addEventListener('input', (e) => {
    const rgb = hexToRgb(e.target.value);
    if (rgb) {
      state.primaryColor = { ...rgb, a: state.primaryColor.a };
      updateColorDisplay();
    }
  });

  elements.hexInput.addEventListener('change', (e) => {
    const rgb = hexToRgb(e.target.value);
    if (rgb) {
      state.primaryColor = { ...rgb, a: state.primaryColor.a };
      updateColorDisplay();
    }
  });

  ['r', 'g', 'b'].forEach(channel => {
    document.getElementById(`${channel}-input`).addEventListener('change', (e) => {
      state.primaryColor[channel] = Math.max(0, Math.min(255, parseInt(e.target.value) || 0));
      updateColorDisplay();
    });
  });

  elements.alphaInput.addEventListener('input', (e) => {
    state.primaryColor.a = parseInt(e.target.value);
    elements.alphaDisplay.textContent = e.target.value;
  });

  elements.primaryColor.addEventListener('click', () => elements.colorPicker.click());

  // Brush size
  elements.brushSize.addEventListener('input', (e) => {
    state.brushSize = parseInt(e.target.value);
    elements.brushSizeDisplay.textContent = e.target.value;
  });

  // Palette presets
  elements.palettePresets.addEventListener('change', (e) => {
    loadPalette(e.target.value);
  });

  // New canvas modal
  document.getElementById('close-new-modal').addEventListener('click', hideNewCanvasModal);
  document.getElementById('cancel-new').addEventListener('click', hideNewCanvasModal);
  document.getElementById('confirm-new').addEventListener('click', createNewCanvas);

  document.getElementById('new-bg').addEventListener('change', (e) => {
    document.getElementById('new-bg-color').style.display = e.target.value === 'custom' ? 'block' : 'none';
  });

  document.querySelectorAll('.preset-btn').forEach(btn => {
    btn.addEventListener('click', () => {
      const size = parseInt(btn.dataset.size);
      document.getElementById('new-width').value = size;
      document.getElementById('new-height').value = size;
    });
  });

  // Shortcuts modal
  document.getElementById('close-shortcuts-modal').addEventListener('click', () => {
    document.getElementById('shortcuts-modal').classList.remove('active');
  });
  document.getElementById('close-shortcuts').addEventListener('click', () => {
    document.getElementById('shortcuts-modal').classList.remove('active');
  });

  // Keyboard shortcuts
  document.addEventListener('keydown', handleKeyDown);

  // IPC events from menu
  ipcRenderer.on('menu-new', showNewCanvasModal);
  ipcRenderer.on('menu-save', saveProject);
  ipcRenderer.on('menu-export', exportPNG);
  ipcRenderer.on('menu-undo', undo);
  ipcRenderer.on('menu-redo', redo);
  ipcRenderer.on('menu-clear', () => {
    saveState();
    const layer = state.layers[state.activeLayerIndex];
    layer.ctx.clearRect(0, 0, state.canvasWidth, state.canvasHeight);
    compositeCanvas();
    renderLayersList();
  });
  ipcRenderer.on('menu-zoom-in', zoomIn);
  ipcRenderer.on('menu-zoom-out', zoomOut);
  ipcRenderer.on('menu-zoom-reset', resetZoom);
  ipcRenderer.on('menu-toggle-grid', toggleGrid);
  ipcRenderer.on('menu-shortcuts', () => {
    document.getElementById('shortcuts-modal').classList.add('active');
  });

  ipcRenderer.on('load-project', (event, projectData) => {
    loadProject(projectData);
  });

  ipcRenderer.on('load-image', (event, filePath) => {
    loadImage(filePath);
  });
}

function handleMouseDown(e) {
  const { x, y } = getPixelCoords(e);
  if (!isInBounds(x, y)) return;

  const isRightClick = e.button === 2;
  const color = isRightClick ? state.secondaryColor : state.primaryColor;
  const layer = state.layers[state.activeLayerIndex];

  state.isDrawing = true;
  state.lastX = x;
  state.lastY = y;

  switch (state.currentTool) {
    case 'pencil':
      saveState();
      drawBrush(layer.ctx, x, y, color);
      compositeCanvas();
      renderLayersList();
      break;

    case 'eraser':
      saveState();
      drawBrush(layer.ctx, x, y, color, true);
      compositeCanvas();
      renderLayersList();
      break;

    case 'fill':
      saveState();
      floodFill(x, y, color);
      compositeCanvas();
      renderLayersList();
      break;

    case 'eyedropper':
      eyedropper(x, y);
      break;

    case 'line':
    case 'rectangle':
    case 'ellipse':
      saveState();
      state.shapeStart = { x, y };
      break;

    case 'select':
      startSelection(x, y);
      break;

    case 'move':
      if (state.selection) {
        copySelection();
        deleteSelection();
      }
      break;
  }
}

function handleMouseMove(e) {
  const { x, y } = getPixelCoords(e);
  elements.statusCoords.textContent = `${Math.max(0, Math.min(x, state.canvasWidth - 1))}, ${Math.max(0, Math.min(y, state.canvasHeight - 1))}`;

  if (!state.isDrawing) return;

  const layer = state.layers[state.activeLayerIndex];
  const isRightClick = e.buttons === 2;
  const color = isRightClick ? state.secondaryColor : state.primaryColor;

  switch (state.currentTool) {
    case 'pencil':
      drawLine(layer.ctx, state.lastX, state.lastY, x, y, color);
      compositeCanvas();
      renderLayersList();
      state.lastX = x;
      state.lastY = y;
      break;

    case 'eraser':
      drawLine(layer.ctx, state.lastX, state.lastY, x, y, color, true);
      compositeCanvas();
      renderLayersList();
      state.lastX = x;
      state.lastY = y;
      break;

    case 'line':
    case 'rectangle':
    case 'ellipse':
      drawPreview(x, y);
      break;

    case 'select':
      updateSelection(x, y);
      break;

    case 'move':
      if (state.selectionData) {
        clearPreview();
        previewCtx.putImageData(state.selectionData.imageData,
          x - Math.floor(state.selectionData.width / 2),
          y - Math.floor(state.selectionData.height / 2));
        state.selectionData.x = x - Math.floor(state.selectionData.width / 2);
        state.selectionData.y = y - Math.floor(state.selectionData.height / 2);
      }
      break;
  }
}

function handleMouseUp(e) {
  if (!state.isDrawing) return;

  const { x, y } = getPixelCoords(e);
  const layer = state.layers[state.activeLayerIndex];
  const color = state.primaryColor;

  switch (state.currentTool) {
    case 'line':
      if (state.shapeStart) {
        drawLine(layer.ctx, state.shapeStart.x, state.shapeStart.y, x, y, color);
        compositeCanvas();
        renderLayersList();
      }
      break;

    case 'rectangle':
      if (state.shapeStart) {
        drawRectangle(layer.ctx, state.shapeStart.x, state.shapeStart.y, x, y, color, e.shiftKey);
        compositeCanvas();
        renderLayersList();
      }
      break;

    case 'ellipse':
      if (state.shapeStart) {
        drawEllipse(layer.ctx, state.shapeStart.x, state.shapeStart.y, x, y, color, e.shiftKey);
        compositeCanvas();
        renderLayersList();
      }
      break;

    case 'move':
      if (state.selectionData) {
        layer.ctx.putImageData(state.selectionData.imageData, state.selectionData.x, state.selectionData.y);
        compositeCanvas();
        renderLayersList();
        state.selectionData = null;
      }
      break;
  }

  state.isDrawing = false;
  state.shapeStart = null;
  clearPreview();

  if (state.currentTool === 'select' && state.selection) {
    drawSelectionOverlay();
  }
}

function handleWheel(e) {
  if (e.ctrlKey || e.metaKey) {
    e.preventDefault();
    if (e.deltaY < 0) {
      zoomIn();
    } else {
      zoomOut();
    }
  }
}

function handleKeyDown(e) {
  // Don't trigger shortcuts when typing in inputs
  if (e.target.tagName === 'INPUT') return;

  const key = e.key.toLowerCase();

  // Tool shortcuts
  const toolKeys = {
    'p': 'pencil',
    'e': 'eraser',
    'f': 'fill',
    'i': 'eyedropper',
    'l': 'line',
    'r': 'rectangle',
    'o': 'ellipse',
    's': 'select',
    'm': 'move'
  };

  if (toolKeys[key] && !e.ctrlKey && !e.metaKey) {
    selectTool(toolKeys[key]);
    return;
  }

  // Other shortcuts
  if (key === 'x' && !e.ctrlKey && !e.metaKey) {
    swapColors();
  } else if (key === 'g' && !e.ctrlKey && !e.metaKey) {
    toggleGrid();
  } else if (key === '[') {
    state.brushSize = Math.max(1, state.brushSize - 1);
    elements.brushSize.value = state.brushSize;
    elements.brushSizeDisplay.textContent = state.brushSize;
  } else if (key === ']') {
    state.brushSize = Math.min(16, state.brushSize + 1);
    elements.brushSize.value = state.brushSize;
    elements.brushSizeDisplay.textContent = state.brushSize;
  } else if (key === '=' || key === '+') {
    zoomIn();
  } else if (key === '-') {
    zoomOut();
  } else if (key === '0' && !e.ctrlKey && !e.metaKey) {
    resetZoom();
  } else if (key === 'delete' || key === 'backspace') {
    if (state.selection) {
      deleteSelection();
      state.selection = null;
      clearPreview();
    }
  } else if (key === 'z' && (e.ctrlKey || e.metaKey)) {
    if (e.shiftKey) {
      redo();
    } else {
      undo();
    }
    e.preventDefault();
  } else if (key === 'n' && (e.ctrlKey || e.metaKey)) {
    e.preventDefault();
    showNewCanvasModal();
  } else if (key === 's' && (e.ctrlKey || e.metaKey)) {
    e.preventDefault();
    saveProject();
  } else if (key === 'e' && (e.ctrlKey || e.metaKey)) {
    e.preventDefault();
    exportPNG();
  }
}

function loadProject(projectData) {
  state.canvasWidth = projectData.canvasWidth;
  state.canvasHeight = projectData.canvasHeight;

  setupCanvas(projectData.canvasWidth, projectData.canvasHeight);

  state.layers = [];
  projectData.layers.forEach((layerData, index) => {
    const canvas = document.createElement('canvas');
    canvas.width = state.canvasWidth;
    canvas.height = state.canvasHeight;
    const ctx = canvas.getContext('2d', { willReadFrequently: true });

    const img = new Image();
    img.onload = () => {
      ctx.drawImage(img, 0, 0);
      if (index === projectData.layers.length - 1) {
        compositeCanvas();
        renderLayersList();
      }
    };
    img.src = layerData.data;

    state.layers.push({
      name: layerData.name,
      canvas,
      ctx,
      visible: layerData.visible,
      opacity: layerData.opacity
    });
  });

  state.activeLayerIndex = projectData.activeLayerIndex;

  // Load palette
  if (projectData.palette) {
    elements.paletteGrid.innerHTML = '';
    projectData.palette.forEach(color => {
      const colorEl = document.createElement('div');
      colorEl.className = 'palette-color';
      colorEl.style.background = color;
      colorEl.dataset.color = color;
      colorEl.addEventListener('click', () => {
        const rgb = hexToRgb(color);
        if (rgb) {
          state.primaryColor = { ...rgb, a: state.primaryColor.a };
          updateColorDisplay();
        }
      });
      elements.paletteGrid.appendChild(colorEl);
    });
  }

  state.undoStack = [];
  state.redoStack = [];
  state.modified = false;
}

function loadImage(filePath) {
  const img = new Image();
  img.onload = () => {
    state.layers = [];
    state.undoStack = [];
    state.redoStack = [];

    setupCanvas(img.width, img.height);
    addLayer('Imported Image');

    const layer = state.layers[0];
    layer.ctx.drawImage(img, 0, 0);
    compositeCanvas();
    renderLayersList();
  };
  img.src = filePath;
}

// Initialize the app
init();

// Set grid button as active initially
document.getElementById('grid-btn').classList.add('active');
